using System;
using System.IO;
using System.Net;
using System.Xml;
using System.Collections;
using System.Net.Sockets;
using System.Diagnostics;

using Bertaccini.Utils;

namespace Team_Project.PersistencyManagers
{
	/// <summary>
	/// Contiene gli argomenti utilizzati per gli eventi scatenati dal treeManager
	/// </summary>
	public class TPCopyEventArgs : EventArgs
	{
		/// <summary>
		/// Nome della copia alla quale l'evento si riferisce 
		/// </summary>
		public string CopyName;
		/// <summary>
		/// Crea gli argomenti dell'evento
		/// </summary>
		/// <param name="name">Nome della copia alla quale si riferisce l'evento</param>
		public TPCopyEventArgs(string name)
		{
			CopyName = name;
		}
	}

	/// <summary>
	/// Delegato utilizzato per gli eventi scatenati dal treeManager
	/// </summary>
	public delegate void CopyEventDelegate(object source,TPCopyEventArgs args);

	/// <summary>
	/// Gestore delle copie strutturate ad albero. Mantiene in memoria 
	/// e in un file la situazione
	/// aggiornata delle copie disponibili e delle loro posizioni relative
	/// </summary>
	/// <remarks>L'istanza globale di questa classe  disponibile con il nome
	/// "TreeManager"</remarks>
	[TeamProjectInit("TreeManager")]
	public class CopiesTreeManager : MarshalByRefObject
	{
		/// <summary>
		/// Nome del file delle copie originale letto all'avvio del servizio
		/// </summary>
		protected string OriginalFileName;
		/// <summary>
		/// Nome del file dove  "fotografata" la situazione attuale delle copie
		/// </summary>
		protected string WorkingFileFullName;
		/// <summary>
		/// Nome della copia locale
		/// </summary>
		protected string LocalName;
		/// <summary>
		/// Nodo che rappresenta la copia locale
		/// </summary>
		protected XmlNode thisCopyNode;
		/// <summary>
		/// Evento scatenato quando una copia viene rimossa dall'albero
		/// </summary>
		public event CopyEventDelegate CopyRemoved;
		/// <summary>
		/// Evento scatenato quando una copia viene aggiunta all'albero
		/// </summary>
		public event CopyEventDelegate CopyAdded;

		/// <summary>
		/// Costruisce un'istanza del gestore delle copie
		/// </summary>
		/// <param name="localCopyName">Nome della copia locale</param>
		public CopiesTreeManager(string localCopyName)
#if DEBUG
			:this(@"..\..\..\CopiesTree.xml",localCopyName)
#else
			:this(@"CopiesTree.xml",localCopyName)
#endif
		{}

		/// <summary>
		/// Inizializza l'istanza visibile globalmente. Questa funzione 
		/// chiamata all'avvio del serivzio dal'Engine
		/// </summary>
		/// <returns>Una copia configurata di CopiesTreeManager</returns>
		public static CopiesTreeManager ComponentInit()
		{
			MessageListener llt = (MessageListener)Globals.Instance.Data["MessageListener"];
			llt.RegisterListener("TM_CP_REMOVE",typeof(TreeMessageListener));
			llt.RegisterListener("TM_ADD_ME",typeof(TreeMessageListener));
			llt.RegisterListener("TM_ADD_REMOTE",typeof(TreeMessageListener));
			return new CopiesTreeManager(Globals.Instance.LocalCopyName);
		}

		/// <summary>
		/// Costruisce un'istanza del gestore delle copie
		/// </summary>
		/// <param name="file">Percorso completo o relativo del file che contiene
		/// la situazione originale (prevista) delle copie</param>
		/// <param name="localCopyName">Nome della copia locale</param>
		public CopiesTreeManager(string file,string localCopyName)
		{
			OriginalFileName = file;
			LocalName = localCopyName;
			FileInfo f = new FileInfo(OriginalFileName);

			WorkingFileFullName = f.DirectoryName + "\\" + localCopyName + "Tree.xml";
			f.CopyTo(WorkingFileFullName,true);
			Trace.WriteLine("Tree file generated for copy " + localCopyName);
			XmlDocument doc = new XmlDocument();
			doc.Load(WorkingFileFullName);
			thisCopyNode = doc.SelectSingleNode("//"+localCopyName);
			if(thisCopyNode == null)
				throw new Team_Project.Exceptions.TeamProjectException("Wrong copy name: " + localCopyName);
		}

		/// <summary>
		/// Restituisce il nome della copia padre di quella locale.
		/// </summary>
		/// <returns>Nome del padre della copia locale. Se il nodo locale  la root
		/// restituisce la stringa vuota</returns>
		public string GetParent()
		{
			if (thisCopyNode.ParentNode.NodeType == XmlNodeType.Document) return string.Empty;
			else return thisCopyNode.ParentNode.Name;
		}

		/// <summary>
		/// Restituisce i nomi delle copie figlie di quella locale.
		/// </summary>
		/// <returns>Array conenente i nomi delle copie figlie di quella locale.
		/// Se il nodo locale  una foglia restituisce un array vuoto.</returns>
		public string[] GetChildren()
		{
			ArrayList res = new ArrayList();
			//int i = 0;
			foreach(XmlNode n in thisCopyNode.ChildNodes)
			{
				if(n.NodeType == XmlNodeType.Element)
				{
					res.Add(n.Name);
					//i++;
				}
			}
			return (string[]) res.ToArray(typeof(string));
		}

		/// <summary>
		/// Aggiunge una copia all'albero. Questa funzione dev'essere richiamata
		/// quando la richiesta di aggiunta della copia NON proviene da quella locale
		/// </summary>
		/// <param name="toAdd">Nome della copia da aggiungere</param>
		/// <param name="theParent">Padre della copia da aggiungere</param>
		private void AddRemoteChildren(string toAdd,string theParent)
		{
			XmlNode pNode = thisCopyNode.OwnerDocument.SelectSingleNode("//" + theParent);
			if(pNode == null)
			{
				Trace.WriteLine("Trying to add " + toAdd + " to " + theParent +
					". Parent does not exists");
				return;
			}
			XmlNode newNode = thisCopyNode.OwnerDocument.SelectSingleNode(toAdd);
			if(newNode != null)
			{
				Trace.WriteLine("Trying to add " + toAdd + " to " + theParent +
					". Dupplicate name found");
				return;
			}
			newNode = thisCopyNode.OwnerDocument.CreateElement(toAdd);
			pNode.AppendChild(newNode);
			thisCopyNode.OwnerDocument.Save(WorkingFileFullName);
		}

		/// <summary>
		/// Aggiunge una copia figlia a quella locale
		/// </summary>
		/// <param name="toAdd">Nome della copia da aggiungere</param>
		public void AddChild(string toAdd)
		{
			XmlNode newNode = thisCopyNode.OwnerDocument.SelectSingleNode(toAdd);
			if(newNode != null)
			{
				Trace.WriteLine("Trying to add " + toAdd + " to this copy." +
					"Dupplicate name found");
				return;
			}
			newNode = thisCopyNode.OwnerDocument.CreateElement(toAdd);
			thisCopyNode.AppendChild(newNode);
			thisCopyNode.OwnerDocument.Save(WorkingFileFullName);
			if(CopyAdded != null)
				CopyAdded(this,new TPCopyEventArgs(toAdd));
		}

		/// <summary>
		/// Rimuove una copia dall'albero
		/// </summary>
		/// <param name="toRemove">Nome della copia da rimuovere</param>
		public void RemoveCopy(string toRemove)
		{
			XmlNode trN = thisCopyNode.SelectSingleNode("//"+toRemove);
			if(trN == null) return;
			XmlDocument doc = trN.OwnerDocument;
			if(trN.ParentNode.NodeType == XmlNodeType.Document) //Dobbiamo rimuovere la root
			{	//Il primo figlio diventa la nuova radice e aggiunge ai propri figli tutti
				// i prpri fratelli
				for(;trN.ChildNodes.Count>1;)
				{
					XmlNode toMove = trN.ChildNodes[1];
					trN.RemoveChild(toMove);
					trN.ChildNodes[0].AppendChild(toMove);
				}
				XmlNode newRoot = trN.ChildNodes[0];
				doc.RemoveAll();
				doc.AppendChild(doc.CreateXmlDeclaration("1.0","UTF-8",""));
				doc.AppendChild(newRoot);
			}
			else // Il parent aggiunge ai propri tutti i figli del nodo da rimuovere
			{
				for(;trN.ChildNodes.Count > 0;)
				{
					XmlNode toMove = trN.ChildNodes[0];
					trN.RemoveChild(toMove);
					trN.ParentNode.AppendChild(toMove);
				}
				trN.ParentNode.RemoveChild(trN);
			}
			doc.Save(WorkingFileFullName);
			if(CopyRemoved != null)
				CopyRemoved(this,new TPCopyEventArgs(toRemove));
			ProcessRemoveCopyProtocol(toRemove);
		}

		/// <summary>
		/// Effettua l'invio dei messaggi di notifica del fatto che una copia
		///  stata eliminata a tutte le copie
		/// </summary>
		/// <param name="name">Nome della copia eliminata</param>
		protected virtual void ProcessRemoveCopyProtocol(string name)
		{
			string s = this.GetParent();
			if(s != null && s != string.Empty)
				SendRemove(s,name);
			foreach(string s2 in this.GetChildren())
				SendRemove(s2,name);
		}

		/// <summary>
		/// Effettua l'invio del messaggio di eliminazione ad una copia
		/// </summary>
		/// <param name="to">Copia alla quale inviare il messaggio</param>
		/// <param name="who">Nome della copia rimossa</param>
		protected virtual void SendRemove(string to,string who)
		{
			Socket s = new Socket(AddressFamily.InterNetwork,SocketType.Stream,ProtocolType.Tcp);
			try
			{
				s.Connect(new IPEndPoint(MyDNS.ResolveIP(to),MyDNS.ListenerPort(to)));
				StreamWriter sw = new StreamWriter(new NetworkStream(s));
				sw.WriteLine("TM_CP_REMOVE");
				sw.WriteLine(who);
				sw.Close();
				s.Shutdown(SocketShutdown.Both);
				s.Close();
			}
			catch(Exception e)
			{
				Trace.WriteLine("SendRemove TO=" + to + " WHO=" + who + " exception:" + e.Message);
			}
		}

		/// <summary>
		/// Confronta la priorit relativa tra due copie
		/// </summary>
		/// <param name="copyName1">Nome della prima copia</param>
		/// <param name="copyName2">Nome della seconda copia</param>
		/// <returns>Un numero positivo se n1  prioritario di n2,
		///  un numero negativo altrimenti.</returns>
		public int ComparePriority(string copyName1,string copyName2)
		{
			XmlNode n1 = thisCopyNode.SelectSingleNode("//"+copyName1);
			XmlNode n2 = thisCopyNode.SelectSingleNode("//"+copyName2);
			int Lvl1 = LevelOf(n1);
			int Lvl2 = LevelOf(n2);
			if (Lvl1 != Lvl2) return Lvl1 - Lvl2;
			XmlNodeOrder no = RelativePosition(n1,n2);
			if(no == XmlNodeOrder.Before) return 1;
			else return -1;
		}

		/// <summary>
		/// Ottiene il livello di profondit dell'albero alla quale si trova un nodo.
		/// </summary>
		/// <remarks>La radice corrisponde al livello 1</remarks>
		/// <param name="n">Nodo del quale calcolare il livello</param>
		/// <returns>Il livello di profondit del nodo</returns>
		protected static int LevelOf(XmlNode n)
		{
			int level = 0;
			XmlNode Ni = n;
			while (Ni.NodeType != XmlNodeType.Document)
			{
				level++;
				Ni = Ni.ParentNode;
			}
			return level;
		}

		/// <summary>
		/// Calcola la posizione relativa di due nodi.
		/// </summary>
		/// <param name="n1">Primo nodo</param>
		/// <param name="n2">Secondo nodo</param>
		/// <returns>
		/// <para>Unknown se i nodi non sono dello stesso livello o appartengono
		/// a documenti diversi</para>
		/// <para>Same se n1 e n2 sono riferimenti allo stesso nodo</para>
		/// <para>Before se n1, in una rappresentazione testuale dell'xml,
		///  scritto prima di n2</para>
		/// <para>After se n2, in una rappresentazione testuale dell'xml,
		///  scritto prima di n1</para>
		/// </returns>
		protected static XmlNodeOrder RelativePosition(XmlNode n1,XmlNode n2)
		{
			if(LevelOf(n1) != LevelOf(n2)) return XmlNodeOrder.Unknown;
			if(n1.OwnerDocument != n2.OwnerDocument) return XmlNodeOrder.Unknown;
			if(n1 == n2) return XmlNodeOrder.Same;
			
			while(n1.ParentNode != n2.ParentNode)
			{
				n1 = n1.ParentNode;
				n2 = n2.ParentNode;
			}
			XmlNode p = n1.ParentNode;
			for(int i = 0; i < p.ChildNodes.Count;i++)
			{
				if(p.ChildNodes[i] == n1) return XmlNodeOrder.Before;
				if(p.ChildNodes[i] == n2) return XmlNodeOrder.After;
			}
			return XmlNodeOrder.Unknown;
		}

		private string SearchOrigialFile()
		{
			XmlDocument orig = new XmlDocument();
			orig.Load(OriginalFileName);
			XmlNode nd = orig.SelectSingleNode("//" + thisCopyNode.Name);
			if(nd == null) return null;
			XmlNode pn = nd.ParentNode;
			if(pn.NodeType != XmlNodeType.Document)
				return pn.Name;
			if(nd.ChildNodes.Count == 0) return null;
			else return nd.ChildNodes[0].Name;
		}
		/// <summary>
		/// Scatena il protocollo necessario per entrare o ri-entrare nell'albero
		/// delle copie di un sistema attivo.
		/// </summary>
		public void AccessTeamProjectTree()
		{
			string parnt = GetParent();
			if(parnt == null || parnt == "")
			{
				string[] ch = GetChildren();
				if(ch.Length == 0)
				{
					parnt = SearchOrigialFile();
					if(parnt == null)
					{
						Trace.WriteLine("TCM: No known copies, cannot access the tree");
						return;
					}
				}
				else
					parnt = ch[0];
			}
			Socket s = new Socket(AddressFamily.InterNetwork,SocketType.Stream,ProtocolType.Tcp);
			try
			{
				s.Connect(new IPEndPoint(MyDNS.ResolveIP(parnt),MyDNS.ListenerPort(parnt)));
				NetworkStream ns = new NetworkStream(s);
				StreamWriter sw = new StreamWriter(ns);
				sw.WriteLine("TM_ADD_ME");
				sw.WriteLine(Globals.Instance.LocalCopyName);
				sw.Flush();
				System.Runtime.Serialization.Formatters.Soap.SoapFormatter sf =
					new System.Runtime.Serialization.Formatters.Soap.SoapFormatter();
				TreeElement teDoc = (TreeElement)sf.Deserialize(ns);
				this.Tree = teDoc.ToXmlDocument();
				sw.Close();
				s.Shutdown(SocketShutdown.Both);
				s.Close();
			}
			catch(Exception e)
			{
				Trace.WriteLine("Trying to acces tree TO=" + parnt + ".Exception:" + e.Message);
			}
		}
		
		private XmlDocument Tree
		{
			get
			{
				return thisCopyNode.OwnerDocument;
			}
			set
			{
				if(CopyRemoved != null)
				{
					string s = GetParent();
					if(s != null && s != string.Empty)
						CopyRemoved(this,new TPCopyEventArgs(s));
					foreach(string ss in GetChildren())
					{
						CopyRemoved(this,new TPCopyEventArgs(ss));
					}
				}
				thisCopyNode = value.SelectSingleNode("//" + Globals.Instance.LocalCopyName);
				if(thisCopyNode == null)
					throw new Team_Project.Exceptions.TeamProjectException("Wrong copy name: " + Globals.Instance.LocalCopyName);
				if(CopyAdded != null)
				{
					string s = GetParent();
					if(s != null && s != string.Empty)
						CopyAdded(this,new TPCopyEventArgs(s));
					foreach(string ss in GetChildren())
					{
						CopyAdded(this,new TPCopyEventArgs(ss));
					}
				}
			}
		}


		#region Listener
		/// <summary>
		/// Classe adibita all'ascolto dei messaggi relativi ai cambiamenti
		/// della struttura dell'albero delle copie
		/// </summary>
		/// <remarks>
		/// Questa classe gestisce i messaggi di tipo:
		/// <code>
		/// TM_CP_REMOVE
		/// TM_ADD_ME
		/// TM_ADD_REMOTE
		/// </code>
		/// </remarks>
		public class TreeMessageListener : IMessageHandler
		{
			/// <summary>
			/// Socket collegata al messaggio che l'istanza sta gestendo
			/// </summary>
			protected Socket skt;
			/// <summary>
			/// Stream di lettura del messaggio che l'istanza sta gestendo
			/// </summary>
			protected NetworkStream strm;
			/// <summary>
			/// Identificativo del tipo di messaggio che l'istanza sta gestendo
			/// </summary>
			protected string message;
			/// <summary>
			/// Reader utilizzato per la lettura del messaggio
			/// </summary>
			protected StreamReader sr;
			/// <summary>
			/// <para>Richiesto dal gestore dei messaggi</para>
			/// <para>Costruisce un'istanza del gestore del messaggio</para>
			/// </summary>
			/// <param name="msg">Identificativo del tipo di messaggio</param>
			/// <param name="s">Socket associata alla comunicazione</param>
			/// <param name="ns">Stream associato alla socket</param>
			public TreeMessageListener(string msg,Socket s,NetworkStream ns)
			{
				message = msg;
				skt = s;
				strm = ns;
				sr = new StreamReader(strm);
			}

			#region IMessageHandler Members
			/// <summary>
			/// Gestisce il messaggio in maniera sincrona
			/// </summary>
			public void Go()
			{
				switch(message)
				{
					case "TM_CP_REMOVE":
						RemoveCopy();
						break;
					case "TM_ADD_ME":
						AddToLocal();
						break;
					case "TM_ADD_REMOTE":
						AddToRemote();
						break;
				}
				skt.Shutdown(SocketShutdown.Both);
				skt.Close();
			}
			#endregion
			private void SendAddRemote(string to,string who,string parent)
			{
				Socket s = new Socket(AddressFamily.InterNetwork,SocketType.Stream,ProtocolType.Tcp);
				try
				{
					s.Connect(new IPEndPoint(MyDNS.ResolveIP(to),MyDNS.ListenerPort(to)));
					StreamWriter sw = new StreamWriter(new NetworkStream(s));
					sw.WriteLine("TM_ADD_REMOTE");
					sw.WriteLine(who);
					sw.WriteLine(parent);
					sw.WriteLine(Globals.Instance.LocalCopyName);
					sw.Close();
					s.Shutdown(SocketShutdown.Both);
					s.Close();
				}
				catch(Exception e)
				{
					Trace.WriteLine("SendAddRemote TO=" + to + " WHO=" + who + " exception:" + e.Message);
				}
			}

			private void RemoveCopy()
			{
				string cpN = sr.ReadLine();
				CopiesTreeManager ctm = (CopiesTreeManager)Globals.Instance.Data["TreeManager"];
				ctm.RemoveCopy(cpN);
			}

			private void AddToLocal()
			{
				Trace.WriteLine("TM_ADD_ME received");
				string cpN = sr.ReadLine();
				Trace.WriteLine("TM_ADD_ME is from " + cpN);
				CopiesTreeManager ctm = (CopiesTreeManager)Globals.Instance.Data["TreeManager"];
				string[] children = ctm.GetChildren();
				ctm.AddChild(cpN);
				string parent = ctm.GetParent();
				if(parent != null && parent != "")
				{
					Trace.WriteLine("Comunicating access of " + cpN + " to " + parent);
					SendAddRemote(parent,cpN,Globals.Instance.LocalCopyName);
				}
				foreach(string chN in children)
				{
					if(chN == cpN) continue;
					Trace.WriteLine("Comunicating access of " + cpN + " to " + chN);
					SendAddRemote(chN,cpN,Globals.Instance.LocalCopyName);
				}
				System.Runtime.Serialization.Formatters.Soap.SoapFormatter sf = 
					new System.Runtime.Serialization.Formatters.Soap.SoapFormatter();
				sf.Serialize(strm,TreeElement.FromXmlNode(ctm.Tree.DocumentElement));
				strm.Flush();
			}

			private void AddToRemote()
			{
				string cpN = sr.ReadLine();
				string pNode = sr.ReadLine();
				string fromNode = sr.ReadLine();
				CopiesTreeManager ctm = (CopiesTreeManager)Globals.Instance.Data["TreeManager"];
				ctm.AddRemoteChildren(cpN,pNode);
				string[] children = ctm.GetChildren();
				string parent = ctm.GetParent();
				if(parent != null && parent != "" && parent != fromNode)
				{
					SendAddRemote(parent,cpN,pNode);
				}
				foreach(string chN in children)
				{
					if(chN == fromNode) continue;
					SendAddRemote(chN,cpN,pNode);
				}
			}
		}
		#endregion
	}
}
